<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Vote;
use App\Models\Candidate;
use App\Models\AdminLog;
use Illuminate\Support\Facades\Storage;

class AdminController extends Controller
{
    // --- LOGIN (Tetap Sama) ---
    public function login(Request $request)
    {
        $correctPin = env('ADMIN_PIN', '1207'); 
        if ($request->pin === $correctPin) {
            AdminLog::create(['activity' => 'Admin berhasil login']);
            $token = bin2hex(random_bytes(16)); 
            return response()->json(['status' => 'success', 'token' => $token]);
        }
        return response()->json(['message' => 'PIN Salah'], 401);
    }

    // --- DASHBOARD (DIUPDATE: Tambah ID di Audit Log) ---
    public function dashboard()
    {
        $candidates = Candidate::withCount('votes')->get();
        
        $chartData = [
            'labels' => $candidates->pluck('name'),
            'data' => $candidates->pluck('votes_count'),
            'total_votes' => Vote::count()
        ];

        $auditLogs = Vote::with(['house', 'candidate'])
            ->orderBy('voted_at', 'desc')
            ->take(50)
            ->get()
            ->map(function($vote) {
                return [
                    'id' => $vote->id, // <--- PENTING: ID DITAMBAHKAN AGAR BISA DIHAPUS
                    'voter_name' => $vote->voter_name,
                    'address' => "Blok " . $vote->house->block . " No " . $vote->house->number,
                    'candidate' => $vote->candidate->name,
                    'time' => date('H:i d M Y', strtotime($vote->voted_at))
                ];
            });

        // Ambil ID juga untuk log admin
        $adminLogs = AdminLog::latest()->take(50)->get();

        return response()->json([
            'chart' => $chartData,
            'audit_logs' => $auditLogs,
            'admin_logs' => $adminLogs
        ]);
    }

    // --- CRUD KANDIDAT (Tetap Sama) ---
    public function storeCandidate(Request $request)
    {
        $request->validate(['name' => 'required', 'visi_misi' => 'required', 'photo' => 'nullable|image|max:2048']);
        $photoUrl = null;
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('candidates', 'public');
            $photoUrl = '/storage/' . $path;
        }
        Candidate::create(['name' => $request->name, 'visi_misi' => $request->visi_misi, 'photo_url' => $photoUrl]);
        AdminLog::create(['activity' => "Menambah kandidat: {$request->name}"]);
        return response()->json(['message' => 'Kandidat berhasil ditambahkan']);
    }

    public function updateCandidate(Request $request, $id)
    {
        $candidate = Candidate::find($id);
        if (!$candidate) return response()->json(['message' => 'Not Found'], 404);
        
        if ($request->hasFile('photo')) {
            if ($candidate->photo_url) {
                $oldPath = str_replace('/storage/', '', $candidate->photo_url);
                Storage::disk('public')->delete($oldPath);
            }
            $path = $request->file('photo')->store('candidates', 'public');
            $candidate->photo_url = '/storage/' . $path;
        }
        $candidate->name = $request->name;
        $candidate->visi_misi = $request->visi_misi;
        $candidate->save();
        AdminLog::create(['activity' => "Mengupdate kandidat: {$candidate->name}"]);
        return response()->json(['message' => 'Data kandidat diperbarui']);
    }

    public function deleteCandidate($id)
    {
        $candidate = Candidate::find($id);
        if (!$candidate) return response()->json(['message' => 'Not Found'], 404);
        if ($candidate->photo_url) {
            Storage::disk('public')->delete(str_replace('/storage/', '', $candidate->photo_url));
        }
        $candidate->delete();
        Vote::where('candidate_id', $id)->delete();
        AdminLog::create(['activity' => "Menghapus kandidat: {$candidate->name}"]);
        return response()->json(['message' => 'Kandidat dihapus']);
    }

    public function resetData()
    {
        Vote::truncate();
        AdminLog::create(['activity' => 'Admin mereset semua suara (NOL)']);
        return response()->json(['message' => 'Semua suara berhasil direset ke 0']);
    }

    // --- FITUR BARU: HAPUS LOG & VOTE ---

    // 1. Hapus Vote (Jika salah input)
    public function deleteVote($id)
    {
        $vote = Vote::find($id);
        if($vote) {
            $voterName = $vote->voter_name;
            $vote->delete();
            // Catat siapa yang menghapus
            AdminLog::create(['activity' => "Menghapus suara milik: {$voterName}"]);
            return response()->json(['message' => 'Suara berhasil dihapus']);
        }
        return response()->json(['message' => 'Data tidak ditemukan'], 404);
    }

    // 2. Hapus Log Admin (Bersih-bersih history)
    public function deleteAdminLog($id)
    {
        $log = AdminLog::find($id);
        if($log) {
            $log->delete();
            return response()->json(['message' => 'Log aktivitas dihapus']);
        }
        return response()->json(['message' => 'Data tidak ditemukan'], 404);
    }

    // --- FITUR BARU: EXPORT EXCEL (CSV) ---
    public function exportCsv()
    {
        $fileName = 'rekap_suara_' . date('Y-m-d_H-i') . '.csv';
        $votes = Vote::with(['house', 'candidate'])->orderBy('voted_at', 'desc')->get();

        $headers = [
            "Content-type"        => "text/csv",
            "Content-Disposition" => "attachment; filename=$fileName",
            "Pragma"              => "no-cache",
            "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
            "Expires"             => "0"
        ];

        $columns = ['Waktu', 'Nama Pemilih', 'Alamat', 'Pilihan Kandidat'];

        $callback = function() use($votes, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($votes as $vote) {
                fputcsv($file, [
                    $vote->voted_at,
                    $vote->voter_name,
                    "Blok " . $vote->house->block . " No " . $vote->house->number,
                    $vote->candidate->name
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}